local BC = rawget(_G, "EURY_COOKINGUI") or {}
_G.EURY_COOKINGUI = BC

BC.Options = BC.Options or {}
local OPS = BC.Options

local INGREDIENT_ICON = getTexture("media/textures/context_ingredients.png")
local SEASONING_ICON  = getTexture("media/textures/context_seasonings.png")

local function BuildRecord(evoItem, evorecipe2, playerObj, cookingLvl)
    if not evoItem then return nil end

    ------------------------------------------------
    -- First, classify seasoning vs non-seasoning
    ------------------------------------------------
    local isSeasoning = BC.isSeasoning(evoItem)

    local use = 0
    local h   = evoItem.getHungerChange and evoItem:getHungerChange() or 0

    if isSeasoning and h < 0 then
        -- Seasonings: use HungerChange as remaining "power"
        local absUnits = math.abs(h) * 100
        use = math.floor(absUnits + 0.5)   -- 0 decimal places, rounded

    elseif instanceof(evoItem, "Food") then
        use = ISInventoryPaneContextMenu.getRealEvolvedItemUse(evoItem, evorecipe2, cookingLvl) or 0

    elseif instanceof(evoItem, "DrainableComboItem") and evoItem.getCurrentUses then
        use = evoItem:getCurrentUses() or 0
    end

    local record = {}

    record.item         = evoItem
    record.fullType     = evoItem:getFullType()
    record.displayName  = evoItem:getName()
    record.use          = use
    record.timeToRot    = BC.getTimeLeft(evoItem)
    record.isFood       = instanceof(evoItem, "Food")
    record.scriptItem   = evoItem:getScriptItem()
    record.isSeasoning  = isSeasoning

    -- Poison / bleach flags (for UI + sandbox handling)
    if record.isFood then
        record.isPoison       = evoItem.isPoison and evoItem:isPoison() or false
        record.poisonLevel    = evoItem.getPoisonLevelForRecipe and evoItem:getPoisonLevelForRecipe() or nil
        record.herbalistType  = evoItem.getHerbalistType and evoItem:getHerbalistType() or nil
    else
        record.isPoison = evoItem.isPoison and evoItem:isPoison() or false
    end

    record.type = evoItem:getType()

    BC.log(
        string.format("type=%s | isSeasoning=%s | uses=%s | timeToRot=%s", tostring(record.fullType), tostring(record.isSeasoning), tostring(record.use), tostring(record.timeToRot))
    )
    return record
end

---------------------------------------------------------------
-- Helper: apply sandbox poisoning rules and derive extraInfo
--
-- Returns:
--   evoItemOrNil, extraInfo string
---------------------------------------------------------------

local function FilterPoisonAndLabel(record, baseItem, playerObj)
    local item = record.item
    local extraInfo = ""

    local enablePoisoning = SandboxVars and SandboxVars.EnablePoisoning or 1
    local disablePoison = (enablePoisoning == 2)

    -- Bleach griefing toggle (vanilla: mode 3)
    if enablePoisoning == 3 and item and record.type == "Bleach" then
        disablePoison = true
    end

    -- Food poison labels
    if record.isFood then
        if record.isSeasoning and not record.isPoison then
            -- Spice-only food
        elseif record.poisonLevel and record.isPoison then
            if disablePoison then
                item = nil
            else
                if record.herbalistType and item.getHerbalistType and item:getHerbalistType()
                    and playerObj:isRecipeActuallyKnown("Herbalist") then
                    extraInfo = getText("ContextMenu_EvolvedRecipe_Poison")
                end
            end
        elseif not record.isPoison and not record.isSeasoning then
            -- Non-poison, non-spice food: show (use) count later per-group.
        end
    else
        -- Non-food poison
        if record.isPoison and disablePoison then
            item = nil
        end
    end

    -- Special-case chum: vanilla clears extraInfo
    if item and baseItem and baseItem:getFullType() == "Base.Chum" then
        extraInfo = ""
    end

    return item, extraInfo
end

---------------------------------------------------------------
-- Helper: create a group record in a single place of truth
---------------------------------------------------------------
local function MakeGroup(bucket, bestRec, timeToRot, use, count, extraInfo, kind)
    return {
        bucket     = bucket,
        bestRecord = bestRec,
        timeToRot  = timeToRot or 9999999,
        use        = use or 0,
        count      = count or 1,
        extraInfo  = extraInfo or "",
        label      = bucket.displayName,
        kind       = kind or "main",
    }
end

---------------------------------------------------------------
-- Helper: append "(use)" and "[xN]" suffixes consistently
--
--   extraInfo: base string (may be empty)
--   use:       number or nil
--   count:     integer or nil
--   showQty:   boolean
---------------------------------------------------------------
local function AppendUseAndQuantity(extraInfo, use, count, showQty)
    local txt = extraInfo or ""

    if use and use > 0 then
        if txt ~= "" then
            txt = txt .. " "
        end
        txt = txt .. "(" .. tostring(use) .. ")"
    end

    if showQty and count and count > 1 then
        txt = txt .. " [x" .. tostring(count) .. "]"
    end

    return txt
end

---------------------------------------------------------------
-- Helper: build buckets (seasoningsByType / ingredientsByType)
--
-- Output:
--   {
--     seasonings = { [fullType] = { key, displayName, records = {...} } },
--     ingredients = { [fullType] = { key, displayName, records = {...} } },
--   }
---------------------------------------------------------------

local function BuildBuckets(evorecipe2, baseItem, playerObj, cookingLvl, items)
    if not items or items:isEmpty() then
        return nil
    end

    local seasonings = {}
    local ingredients = {}

    for i = 0, items:size() - 1 do
        local evoItem = items:get(i)
        local rec = BuildRecord(evoItem, evorecipe2, playerObj, cookingLvl)
        if rec then
            local bucketTable = rec.isSeasoning and seasonings or ingredients
            local key = rec.fullType
            local bucket = bucketTable[key]
            if not bucket then
                bucket = {
                    key         = key,
                    displayName = rec.displayName,
                    records     = {},
                }
                bucketTable[key] = bucket
            end
            table.insert(bucket.records, rec)
        end
    end

    return {
        seasonings  = seasonings,
        ingredients = ingredients,
    }
end

---------------------------------------------------------------
-- BuildSeasoningGroups
--
-- Input:
--   buckets   - table of { displayName, records = {record,...} }
--   baseItem  - the base dish item
--   playerObj - IsoPlayer (for poison / sandbox rules)
--
-- Output:
--   groups = {
--       {
--           bucket     = bucket,
--           bestRecord = rec,
--           timeToRot  = number,
--           use        = number,
--           count      = integer used for [xN],
--           extraInfo  = string,
--           label      = bucket.displayName,
--       },
--       ...
--   }
--
-- Behaviour:
--   - Always separates GOOD vs ROTTEN internally.
--   - Builds at most ONE good group and ONE rotten group per bucket.
--   - Good group: spoilage/uses logic as before.
--   - Rotten group: highest use first, earliest rot tiebreak.
--   - Quantity [xN] is never mixed between good/rotten.
---------------------------------------------------------------
local function BuildSeasoningGroups(buckets, baseItem, playerObj)
    local showQty        = OPS.shouldShowQuantity()
    local sortBySpoilage = OPS.shouldSortBySpoilage()
    local sortByUses     = OPS.shouldSortByUsesRemaining()
    local showRotten     = OPS.shouldShowRottenSeasoningGroup()

    local groups = {}
    local itemForTexture = nil

    for _, bucket in pairs(buckets) do
        local goodRecords   = {}
        local rottenRecords = {}

        ------------------------------------------------
        -- Classify records into good vs rotten,
        -- applying poison / sandbox filtering.
        ------------------------------------------------
        for _, rec in ipairs(bucket.records) do
            local item, extraInfo = FilterPoisonAndLabel(rec, baseItem, playerObj)
            if item then
                rec.extraInfoBase = extraInfo

                local evoItem = rec.item
                local isRotten = evoItem.isRotten and evoItem:isRotten() or false

                if isRotten then
                    table.insert(rottenRecords, rec)
                else
                    table.insert(goodRecords, rec)
                end
            end
        end

        ------------------------------------------------
        -- GOOD seasoning group
        ------------------------------------------------
        if #goodRecords > 0 then
            local bestRec  = nil
            local bestTime = nil
            local bestUse  = nil
            local count    = #goodRecords

            for _, rec in ipairs(goodRecords) do
                local t = rec.timeToRot
                local u = rec.use or 0

                if not bestRec then
                    bestRec  = rec
                    bestTime = t
                    bestUse  = u
                else
                    local better = false

                    if sortBySpoilage and sortByUses then
                        -- 1) Spoilage primary
                        if t < bestTime then
                            better = true
                        -- 2) Uses secondary when spoilage ties (least uses first)
                        elseif t == bestTime and u < bestUse then
                            better = true
                        end

                    elseif sortBySpoilage then
                        if t < bestTime then
                            better = true
                        end

                    elseif sortByUses then
                        if u < bestUse then
                            better = true
                        end
                    end

                    if better then
                        bestRec  = rec
                        bestTime = t
                        bestUse  = u
                    end
                end
            end

            if bestRec then
                local extraInfo = bestRec.extraInfoBase or ""

                -- "Spice" label
                if bestRec.isSeasoning then
                    local spiceText = getText("ContextMenu_EvolvedRecipe_Spice")
                    if extraInfo == "" then
                        extraInfo = spiceText
                    else
                        extraInfo = extraInfo .. " " .. spiceText
                    end
                end

                -- Append "(use)" and "[xN]" via helper (no behaviour change)
                extraInfo = AppendUseAndQuantity(extraInfo, bestUse, count, showQty)

                table.insert(groups, MakeGroup(
                    bucket,
                    bestRec,
                    bestTime,
                    bestUse,
                    count,
                    extraInfo,
                    "main"      -- good seasoning
                ))

                -- Prefer a GOOD seasoning as icon candidate
                if not itemForTexture and bestRec.item then
                    itemForTexture = bestRec.item
                end
            end
        end

        ------------------------------------------------
        -- ROTTEN seasoning group (single consolidated)
        ------------------------------------------------
        if showRotten and #rottenRecords > 0 then
            local bestRec  = nil
            local bestTime = nil
            local bestUse  = nil
            local count    = #rottenRecords

            for _, rec in ipairs(rottenRecords) do
                local t = rec.timeToRot
                local u = rec.use or 0

                if not bestRec then
                    bestRec  = rec
                    bestTime = t
                    bestUse  = u
                else
                    local better = false

                    -- Rotten: primary = HIGHEST use, tiebreaker = earliest rot
                    if u > bestUse then
                        better = true
                    elseif u == bestUse and sortBySpoilage and t < bestTime then
                        better = true
                    end

                    if better then
                        bestRec  = rec
                        bestTime = t
                        bestUse  = u
                    end
                end
            end

            if bestRec then
                local extraInfo = bestRec.extraInfoBase or ""

                if bestRec.isSeasoning then
                    local spiceText = getText("ContextMenu_EvolvedRecipe_Spice")
                    if extraInfo == "" then
                        extraInfo = spiceText
                    else
                        extraInfo = extraInfo .. " " .. spiceText
                    end
                end

                extraInfo = AppendUseAndQuantity(extraInfo, bestUse, count, showQty)

                table.insert(groups, MakeGroup(
                    bucket,
                    bestRec,
                    bestTime,
                    bestUse,
                    count,
                    extraInfo,
                    "rotten"    -- rotten seasoning
                ))

                -- Only use rotten as icon if no good candidate exists
                if not itemForTexture and bestRec.item then
                    itemForTexture = bestRec.item
                end
            end
        end
    end

    return groups, itemForTexture
end

---------------------------------------------------------------
-- BuildIngredientGroups
--
-- Splits:
--   - GOOD vs ROTTEN (using evoItem:isRotten())
--   - Within GOOD: FULL vs PARTIAL (by max use)
--
-- Options:
--   - OPS.shouldShowQuantity()
--   - OPS.shouldShowPartialIngredients()
--   - OPS.shouldShowRottenIngredientGroup()
--
-- Output groups follow the same shape as seasoning groups and
-- are consumed by AddIngredientsToMenu.
---------------------------------------------------------------
local function BuildIngredientGroups(buckets, baseItem, playerObj)
    local groups        = {}
    local showQty       = OPS.shouldShowQuantity()
    local showPartial   = OPS.shouldShowPartialIngredients()
    local sortBySpoil   = OPS.shouldSortBySpoilage()
    local sortByUses    = OPS.shouldSortByUsesRemaining()
    local showRotten    = OPS.shouldShowRottenIngredientGroup()

    local itemForTexture = nil

    for _, bucket in pairs(buckets) do
        local goodRecords   = {}
        local rottenRecords = {}

        ------------------------------------------------
        -- Filter + split into GOOD vs ROTTEN
        ------------------------------------------------
        for _, rec in ipairs(bucket.records) do
            local item, extraInfo = FilterPoisonAndLabel(rec, baseItem, playerObj)
            if item then
                rec.extraInfoBase = extraInfo

                local evoItem = rec.item
                local isRotten = evoItem.isRotten and evoItem:isRotten() or false

                if isRotten then
                    table.insert(rottenRecords, rec)
                else
                    table.insert(goodRecords, rec)
                end
            end
        end

        ------------------------------------------------
        -- GOOD side: FULL + PARTIAL
        ------------------------------------------------
        if #goodRecords > 0 then
            -- Compute maxUse for GOOD records only
            local maxUse = 0
            for _, rec in ipairs(goodRecords) do
                local u = rec.use or 0
                if u > maxUse then
                    maxUse = u
                end
            end

            local fullList    = {}
            local partialList = {}

            for _, rec in ipairs(goodRecords) do
                if rec.use == maxUse then
                    table.insert(fullList, rec)
                else
                    table.insert(partialList, rec)
                end
            end

            -- If everything is effectively "partial", treat all as full
            if #fullList == 0 then
                fullList    = goodRecords
                partialList = {}
            end

            local fullCount    = #fullList
            local partialCount = #partialList
            local totalGood    = fullCount + partialCount

            ------------------------------------------------
            -- Main GOOD (full) group
            ------------------------------------------------
            if fullCount > 0 then
                local mainRec  = nil
                local mainTime = nil
                local mainUse  = nil

                for _, rec in ipairs(fullList) do
                    local t = rec.timeToRot
                    local u = rec.use or 0
                    if not mainRec then
                        mainRec  = rec
                        mainTime = t
                        mainUse  = u
                    else
                        local better = false

                        if sortBySpoil and t ~= mainTime then
                            -- Spoilage primary
                            better = (t < mainTime)
                        elseif sortByUses and u ~= mainUse then
                            -- Uses as tiebreaker (least uses first)
                            better = (u < mainUse)
                        end

                        if better then
                            mainRec  = rec
                            mainTime = t
                            mainUse  = u
                        end
                    end
                end

                if mainRec then
                    local extraInfo = mainRec.extraInfoBase or ""

                    -- Quantity logic: choose count first
                    local qtyCount = totalGood
                    if showPartial then
                        qtyCount = fullCount
                    end

                    extraInfo = AppendUseAndQuantity(extraInfo, mainUse, qtyCount, showQty)

                    table.insert(groups, MakeGroup(
                        bucket,
                        mainRec,
                        mainTime,
                        mainUse,
                        qtyCount,
                        extraInfo,
                        "main"     -- good full ingredient
                    ))

                    -- Prefer GOOD full as icon candidate
                    if not itemForTexture and mainRec.item then
                        itemForTexture = mainRec.item
                    end
                end
            end

            ------------------------------------------------
            -- GOOD partial group (optional)
            ------------------------------------------------
            if showPartial and partialCount > 0 then
                local partialBest = nil
                local partialTime = nil
                local partialUse  = nil

                for _, rec in ipairs(partialList) do
                    local t = rec.timeToRot
                    local u = rec.use or 0

                    if not partialBest then
                        partialBest = rec
                        partialTime = t
                        partialUse  = u
                    else
                        local better = false

                        -- Spoilage-first, then least uses
                        if sortBySpoil and t ~= partialTime then
                            better = (t < partialTime)
                        elseif sortByUses and u ~= partialUse then
                            better = (u < partialUse)
                        end

                        if better then
                            partialBest = rec
                            partialTime = t
                            partialUse  = u
                        end
                    end
                end

                if partialBest then
                    local extraInfo = partialBest.extraInfoBase or ""
                    local partialText = getTextOrNull("UI_BetterCooking_Partial") or "Partial"

                    if extraInfo == "" then
                        extraInfo = partialText
                    else
                        extraInfo = extraInfo .. " " .. partialText
                    end

                    extraInfo = AppendUseAndQuantity(extraInfo, partialUse, partialCount, showQty)

                    table.insert(groups, MakeGroup(
                        bucket,
                        partialBest,
                        partialTime,
                        partialUse,
                        partialCount,
                        extraInfo,
                        "partial"  -- good partial ingredient
                    ))

                    -- Fallback: use partial if no full candidate exists
                    if not itemForTexture and partialBest.item then
                        itemForTexture = partialBest.item
                    end
                end
            end
        end

        ------------------------------------------------
        -- ROTTEN side: single consolidated group
        ------------------------------------------------
        if showRotten and #rottenRecords > 0 then
            local bestRec  = nil
            local bestTime = nil
            local bestUse  = nil
            local count    = #rottenRecords

            for _, rec in ipairs(rottenRecords) do
                local t = rec.timeToRot
                local u = rec.use or 0

                if not bestRec then
                    bestRec  = rec
                    bestTime = t
                    bestUse  = u
                else
                    local better = false

                    -- Rotten: primary = HIGHEST use, tiebreaker = earliest rot
                    if u > bestUse then
                        better = true
                    elseif u == bestUse and sortBySpoil and t < bestTime then
                        better = true
                    end

                    if better then
                        bestRec  = rec
                        bestTime = t
                        bestUse  = u
                    end
                end
            end

            if bestRec then
                local extraInfo = bestRec.extraInfoBase or ""
                extraInfo = AppendUseAndQuantity(extraInfo, bestUse, count, showQty)

                table.insert(groups, MakeGroup(
                    bucket,
                    bestRec,
                    bestTime,
                    bestUse,
                    count,
                    extraInfo,
                    "rotten"
                ))

                -- Last resort: rotten as icon, only if nothing else
                if not itemForTexture and bestRec.item then
                    itemForTexture = bestRec.item
                end
            end
        end
    end

    return groups, itemForTexture
end

---------------------------------------------------------------
-- Helper: sort groups according to global options
---------------------------------------------------------------

local function SortGroups(groups)
    table.sort(groups, function(a, b)
        local aLabel = tostring(a.label or "")
        local bLabel = tostring(b.label or "")

        -- 1) Alphabetical by label (type name)
        if aLabel ~= bLabel then
            return aLabel < bLabel
        end

        -- 2) Within same label, order by "kind":
        --    main -> partial -> rotten
        local order = { main = 1, partial = 2, rotten = 3 }
        local aKind = a.kind or "main"
        local bKind = b.kind or "main"
        local aRank = order[aKind] or 99
        local bRank = order[bKind] or 99

        if aRank ~= bRank then
            return aRank < bRank
        end

        -- 3) Stable-ish fallback (don't swap)
        return false
    end)
end

---------------------------------------------------------------
-- Helper: inject Seasonings groups into the menu
---------------------------------------------------------------

local function AddSeasoningsToMenu(context, subMenuRecipe, evorecipe2, baseItem, player, groups, itemForTexture)
    if #groups == 0 then return end

    SortGroups(groups)

    local useSubMenu = OPS.shouldUseSeasoningsSubMenu()
    local parentMenu = subMenuRecipe
    local seasoningsSubMenu = nil

    if useSubMenu then
        local opt = subMenuRecipe:addOption(getTextOrNull("UI_BetterCooking_Seasonings") or "Seasonings", nil)
        opt.iconTexture = SEASONING_ICON
        --if itemForTexture then
        --    opt.itemForTexture = itemForTexture
        --end
        
        -- Must create submenus from the root context, passing the *parent* menu.
        seasoningsSubMenu = context:getNew(subMenuRecipe)
        subMenuRecipe:addSubMenu(opt, seasoningsSubMenu)
        parentMenu = seasoningsSubMenu
    end

    for _, g in ipairs(groups) do
        local rec = g.bestRecord
        local evoItem = rec.item
        local extraInfo = g.extraInfo or ""

        -- Let vanilla build "<ItemName> <extraInfo>" and handle icons/tooltip.
        ISInventoryPaneContextMenu.addItemInEvoRecipe(
            parentMenu,
            baseItem,
            evoItem,
            extraInfo,
            evorecipe2,
            player
        )
    end
end

---------------------------------------------------------------
-- Helper: inject Ingredient groups into the menu
---------------------------------------------------------------

local function AddIngredientsToMenu(context, subMenuRecipe, evorecipe2, baseItem, player, groups, itemForTexture)
    if #groups == 0 then return end

    SortGroups(groups)

    local useSubMenu = OPS.shouldUseIngredientsSubMenu()
    local parentMenu = subMenuRecipe
    local ingredientsSubMenu = nil

    if useSubMenu then
        local opt = subMenuRecipe:addOption(getTextOrNull("UI_BetterCooking_Ingredients") or "Ingredients", nil)
        if itemForTexture then
            opt.itemForTexture = itemForTexture
        end

        -- Same fix: root context builds the submenu, parent is subMenuRecipe.
        ingredientsSubMenu = context:getNew(subMenuRecipe)
        subMenuRecipe:addSubMenu(opt, ingredientsSubMenu)
        parentMenu = ingredientsSubMenu
    end

    for _, g in ipairs(groups) do
        local rec = g.bestRecord
        local evoItem = rec.item
        local extraInfo = g.extraInfo or ""

        ISInventoryPaneContextMenu.addItemInEvoRecipe(
            parentMenu,
            baseItem,
            evoItem,
            extraInfo,
            evorecipe2,
            player
        )
    end
end

local vanilla_doEvorecipeMenu = ISInventoryPaneContextMenu.doEvorecipeMenu
---------------------------------------------------------------
-- Override: ISInventoryPaneContextMenu.doEvorecipeMenu
--
-- This fully replaces vanilla's evolved recipe menu building,
-- but preserves the "Add Random [FoodType]" logic.
---------------------------------------------------------------
local function OnGameStart()
    ISInventoryPaneContextMenu.doEvorecipeMenu = function(context, items, player, evorecipe, baseItem, containerList)
        local playerObj = getSpecificPlayer(player)
        if not playerObj then
            BC.log("No playerObj; falling back to vanilla doEvorecipeMenu")
            return vanilla_doEvorecipeMenu(context, items, player, evorecipe, baseItem, containerList)
        end

        for i = 0, evorecipe:size() - 1 do
            local evorecipe2 = evorecipe:get(i)
            if not evorecipe2 then
                break
            end

            -- First pass: build buckets (seasonings & ingredients) from raw candidates
            local useableItems = evorecipe2:getItemsCanBeUse(playerObj, baseItem, containerList)
            local cookingLvl = playerObj:getPerkLevel(Perks.Cooking)
            local buckets = BuildBuckets(evorecipe2, baseItem, playerObj, cookingLvl, useableItems)
            if not buckets then
                break
            end

            -- Categories for "Add Random [FoodType]" (vanilla logic)
            local catList = ISInventoryPaneContextMenu.getEvoItemCategories(
                useableItems,
                evorecipe2
            )

            -- Recipe submenu root
            local subOption = nil
            local fromName = getText("ContextMenu_EvolvedRecipe_" .. evorecipe2:getUntranslatedName())
            if evorecipe2:isResultItem(baseItem) then
                subOption = context:addOption(fromName, nil)
            else
                local createName = getText("ContextMenu_Create_From_Ingredient", fromName)
                subOption = context:addOption(createName, nil)
            end

            local subMenuRecipe = context:getNew(context)
            context:addSubMenu(subOption, subMenuRecipe)

            -------------------------------------------------------
            -- 1) Add Random [FoodType] entries (unchanged)
            -------------------------------------------------------

            for catName, list in pairs(catList) do
                if getText("ContextMenu_FoodType_" .. catName) ~= "ContextMenu_FoodType_" .. catName then
                    local foodTypeName = getText("ContextMenu_FoodType_" .. catName)
                    local txt = getText("ContextMenu_FromRandom", foodTypeName)
                    if evorecipe2:isResultItem(baseItem) then
                        txt = getText("ContextMenu_AddRandom", foodTypeName)
                    end
                    local randomIndex = ZombRand(1, #list + 1)
                    subMenuRecipe:addOption(
                        txt,
                        evorecipe2,
                        ISInventoryPaneContextMenu.onAddItemInEvoRecipe,
                        baseItem,
                        list[randomIndex],
                        player
                    )
                end
            end

            -------------------------------------------------------
            -- 2) Build & inject Ingredients
            -------------------------------------------------------

            local ingredientGroups, ingredientIconItem = BuildIngredientGroups(buckets.ingredients, baseItem, playerObj)
            AddIngredientsToMenu(context, subMenuRecipe, evorecipe2, baseItem, player, ingredientGroups, ingredientIconItem)

            -------------------------------------------------------
            -- 3) Build & inject Seasonings
            -------------------------------------------------------

            local seasoningGroups, seasoningIconItem = BuildSeasoningGroups(buckets.seasonings, baseItem, playerObj)
            AddSeasoningsToMenu(context, subMenuRecipe, evorecipe2, baseItem, player, seasoningGroups, seasoningIconItem)
        end
    end
end

Events.OnGameStart.Add(OnGameStart)